/* This file implements the synchronization and communication
   algorithms for the game. This first implementation is simple: a
   fixed site is always the coordinator (so the game is stuck if that
   site goes down). The coordinator can be any site; it is set below
   to the referee site. */

#include "game.h"

#define OK 0
#define NOT_OK 1

#define GET_LOCK  10
#define PUT_LOCK  11

#define GET_FIELD 20
#define PUT_FIELD 21

/* In the coordinator: */
static int lock_holder;
static int lock_request_queue[NUM_SITES];
static int queue_start, queue_end;

void find_coordinator()
{
  coordinator = REFEREE_SITE;
  have_lock = (me == coordinator);
}

static void send_recv_msg(int to, 
                          int to_tag, int to_sz, void *to_data, 
                          int from_tag, int from_sz, void *from_data)
{
  int from, tag, sz;
  void *data;

  send_msg(to, to_tag, to_sz, to_data);

  data = recv_msg(&from, &tag, &sz, 3.0);

  if (!data) {
    fprintf(stderr, 
            "didn't get reply to %d\n",
            to_tag);
    abort();
  }

  if (from != to) {
    fprintf(stderr, "expected reply from %d, got from %d\n",
            to, from);
    abort();
  }

  if (tag != from_tag) {
    fprintf(stderr, "expected %d reply, got %d\n",
            from_tag, tag);
    abort();
  }

  if (sz != from_sz) {
    fprintf(stderr, "expected reply for size %d, got %d\n",
            from_sz, sz);
    abort();
  }

  memcpy(from_data, data, sz);
}

static int send_recv_coord_msg(int to_tag, int to_sz, void *to_data, 
                               int from_tag, int from_sz, void *from_data)
{
  /* If this is the coordinator process. */
  if (me == coordinator)
    return have_lock;

  send_recv_msg(coordinator,
                to_tag, to_sz, to_data, 
                from_tag, from_sz, from_data);
  return 1;
}

int get_field_lock()
{
  return send_recv_coord_msg(GET_LOCK,
                             0, NULL,
                             OK,
                             0, NULL);
}

int get_field_state()
{
  return send_recv_coord_msg(GET_FIELD,
                             0, NULL,
                             PUT_FIELD,
                             sizeof(s), &s);
}

int put_field_lock()
{
  return send_recv_coord_msg(PUT_LOCK,
                             0, NULL,
                             OK,
                             0, NULL);
}

int put_field_state()
{
  return send_recv_coord_msg(PUT_FIELD,
                             sizeof(s), &s,
                             OK,
                             0, NULL);
}

static void coordinate()
{
  int from, tag, sz;
  void *data;

  if (have_lock && (queue_start != queue_end)) {
    from = lock_request_queue[queue_start];
    send_msg(from, OK, 0, NULL);
    have_lock = 0;
    lock_holder = from;
    queue_start = (queue_start + 1) % NUM_SITES;
  }

  data = recv_msg(&from, &tag, &sz, 0.0);
  if (!data) return;

  switch (tag) {
  case GET_LOCK:
    if (sz != 0) {
      send_msg(from, NOT_OK, 0, NULL);
    } else if (have_lock) {
      send_msg(from, OK, 0, NULL);
      lock_holder = from;
      have_lock = 0;
    } else {
      lock_request_queue[queue_end] = from;
      queue_end = (queue_end + 1) % NUM_SITES;
    }
    break;
  case PUT_LOCK:
    if (sz != 0) {
      send_msg(from, NOT_OK, 0, NULL);
    } else if (have_lock || (lock_holder != from)) {
      send_msg(from, NOT_OK, 0, NULL);
    } else {
      send_msg(from, OK, 0, NULL);
      have_lock = 1;
    }
    break;
  case GET_FIELD:
    if (sz != 0) {
      send_msg(from, NOT_OK, 0, NULL);
    } else {
      send_msg(from, PUT_FIELD, sizeof(s), &s);
    }
    break;
  case PUT_FIELD:
    if (sz != sizeof(s) || (lock_holder != from)) {
      send_msg(from, NOT_OK, 0, NULL);
    } else {
      memcpy(&s, data, sizeof(s));
      send_msg(from, OK, 0, NULL);
    }
    break;
  default:
    fprintf(stderr, "unexpected message %d\n", tag);
    abort();
  }
}

void cooperate()
{
  if (me == coordinator)
    coordinate();
}
