/* With this implementation of the communication functions, a
   coordinator is determined by the election algorithm (to select the
   site with th elowest site number), so the game can tolerate the
   disapperance of the coordinator. */

#include "game.h"
#include <sys/time.h>

#define OK 0
#define NOT_OK 1

#define GET_LOCK  10
#define PUT_LOCK  11

#define GET_FIELD 20
#define PUT_FIELD 21

#define ELECT      30
#define INAUGURATE 31

#define WAIT_COORD_FAIL 3.0
#define WAIT_ELECT      3.0

/* In the coordinator: */
static int lock_holder;
static int lock_request_queue[NUM_SITES];
static int queue_start, queue_end;

double now()
{
  struct timeval tv;

  (void)gettimeofday(&tv, NULL);

  return (double)tv.tv_sec + (double)tv.tv_usec/10000000.0;
}

static void election()
{
  /* Coordinator failed! */
  double start_time = now();
  int it_is_me = 1, elected = 0;

  fprintf(stderr, "running for election...\n");
  broadcast_msg(ELECT, sizeof(int), &me);
    
  while (!elected) { 
    int from, tag, sz;
    void *data;
    double timeout;

    if (it_is_me) {
      timeout = WAIT_ELECT - (now() - start_time);
      if (timeout <= 0.0)
        break;
    } else
      /* wait for a nomination: */
      timeout = 0.0;
      
    data = recv_msg(&from, &tag, &sz, timeout);
    if (data) {
      switch (tag) {
      case OK:
        if (from == coordinator) {
          /* `from' apparently hadn't head about the election when it
             sent that message*/
        }
        break;
      case GET_LOCK:
      case PUT_LOCK:
      case GET_FIELD:
      case PUT_FIELD:
        if (me == coordinator) {
          /* `from' apparently hadn't head about the election when it
             sent that message */
        } else {
          fprintf(stderr, "got misdirected %d...\n", *(int *)data);
        }
        break;
      case ELECT:
        fprintf(stderr, "got elect %d...\n", *(int *)data);
        if (*(int *)data < me) {
          /* other site should be elected: */
          it_is_me = 0;
          start_time = now();
        } else {
          broadcast_msg(ELECT, sizeof(int), &me);
        }
        break;
      case INAUGURATE:
        /* other site was elected: */
        coordinator = *(int *)data;
        fprintf(stderr, "elected %d\n", coordinator);
        elected = 1;
        break;
      default:
        fprintf(stderr, "unexpected message %d\n", tag);
        abort();
      }
    }
  }
    
  if (it_is_me) {
    fprintf(stderr, "I accept!\n");
    broadcast_msg(INAUGURATE, sizeof(int), &me);
    coordinator = me;
    have_lock = 1;
    queue_start = queue_end = 0;
  } else
    have_lock = 0;
}

void find_coordinator()
{
  election();
}

static int send_recv_msg_or_elect(int to_tag, int to_sz, void *to_data, 
                                  int from_tag, int from_sz, void *from_data)
{
  int from, tag, sz;
  void *data;

  if (me == coordinator)
    return have_lock;

  send_msg(coordinator, to_tag, to_sz, to_data);
  
  data = recv_msg(&from, &tag, &sz, WAIT_COORD_FAIL);

  if (data && (from == coordinator) && (tag == from_tag) && (sz == from_sz)) {
    /* the expected result: */
    memcpy(from_data, data, sz);
    return 1;
  }

  /* unexpected result: start an election */
  election();
  return 0;
}

int get_field_lock()
{
  return send_recv_msg_or_elect(GET_LOCK,
                                0, NULL,
                                OK,
                                0, NULL);
}

int get_field_state()
{
  return send_recv_msg_or_elect(GET_FIELD,
                                0, NULL,
                                PUT_FIELD,
                                sizeof(s), &s);
}

int put_field_lock()
{
  return send_recv_msg_or_elect(PUT_LOCK,
                                0, NULL,
                                OK,
                                0, NULL);
}

int put_field_state()
{
  return send_recv_msg_or_elect(PUT_FIELD,
                                sizeof(s), &s,
                                OK,
                                0, NULL);
}

static void coordinate()
{
  int from, tag, sz;
  void *data;

  if (have_lock && (queue_start != queue_end)) {
    from = lock_request_queue[queue_start];
    send_msg(from, OK, 0, NULL);
    have_lock = 0;
    lock_holder = from;
    queue_start = (queue_start + 1) % NUM_SITES;
  }

  data = recv_msg(&from, &tag, &sz, 1.0);
  if (!data) return;

  switch (tag) {
  case GET_LOCK:
    if (sz != 0) {
      send_msg(from, NOT_OK, 0, NULL);
    } else if (have_lock) {
      send_msg(from, OK, 0, NULL);
      lock_holder = from;
      have_lock = 0;
    } else {
      lock_request_queue[queue_end] = from;
      queue_end = (queue_end + 1) % NUM_SITES;
    }
    break;
  case PUT_LOCK:
    if (sz != 0) {
      send_msg(from, NOT_OK, 0, NULL);
    } else if (have_lock && (lock_holder == from)) {
      send_msg(from, NOT_OK, 0, NULL);
    } else {
      send_msg(from, OK, 0, NULL);
      have_lock = 1;
    }
    break;
  case GET_FIELD:
    if (sz != 0) {
      send_msg(from, NOT_OK, 0, NULL);
    } else {
      send_msg(from, PUT_FIELD, sizeof(s), &s);
    }
    break;
  case PUT_FIELD:
    if (sz != sizeof(s) || (lock_holder != from)) {
      send_msg(from, NOT_OK, 0, NULL);
    } else {
      memcpy(&s, data, sizeof(s));
      send_msg(from, OK, 0, NULL);
    }
    break;
  case ELECT:
    election();
    break;
  default:
    fprintf(stderr, "unexpected message %d\n", tag);
    abort();
  }
}

static void politic()
{
  int from, tag, sz;
  void *data;

  data = recv_msg(&from, &tag, &sz, 0.1);
  if (!data) return;

  switch (tag) {
  case ELECT:
    election();
    break;
  case INAUGURATE:
    coordinator = *(int *)data;
    fprintf(stderr, "new coordinator! %d\n", coordinator);
    break;
  default:
    fprintf(stderr, "unexpected message %d\n", tag);
    abort();
  }
}

void cooperate()
{
  if (me == coordinator)
    coordinate();
  else
    politic();
}
