/* With this implementation of the communication functions, this is no
   one coordinator. A process takes control of the field by getting a
   majority of other sites to grant it control, and then it boradcasts
   the updated boward after it is done. The tricky part is using a
   kind of virtual clock to give each site a chance to control the
   board. */

#include "game.h"
#include <sys/time.h>

#define WHEN_DEBUG(x) /*x*/

#define WANT_STATE    10
#define GRANT_STATE   11
#define RELEASE_STATE 12
#define MAYBE_CANCEL  13

#define HOUR_HAND_INC 0x100
static int vclock = HOUR_HAND_INC;

static int request_vclock[NUM_SITES];
static int granted_out;
static int grant_in_count;

void find_coordinator()
{
  coordinator = me;
  have_lock = 0;
}

int get_field_lock()
{
  return have_lock;
}

int get_field_state()
{
  return have_lock;
}

int put_field_lock()
{
  return have_lock;
}

int put_field_state()
{
  return have_lock;
}

static int lt(int site_a, int vc_a, int site_b, int vc_b)
/* timestamp ordering: first by vclock, then by site number */
{
  if (vc_a < vc_b
      || (vc_a == vc_b && site_a < site_b))
    return 1;
  else
    return 0;
}

static void check_grant() 
{
  /* If we're not holding the lock ourselves, see whether we should be
     giving the lock to any process. */
  if (!have_lock) {
    int i;
    int min_req = 0;

    for (i = 0; i < NUM_SITES; i++) {
      WHEN_DEBUG(fprintf(stderr, "state [%d] = %d\n", i, request_vclock[i]));
      if (request_vclock[i]
          && (!min_req
              || request_vclock[i] < request_vclock[min_req])) {
        min_req = i;
      }
    }
        
    if (min_req && (min_req != me)) {
      if (!granted_out) {
        WHEN_DEBUG(fprintf(stderr, "granted to %d at %d\n", min_req, request_vclock[min_req]));
        granted_out = min_req;
        send_msg(min_req, GRANT_STATE, sizeof(int), &request_vclock[min_req]);
      } else if (lt(min_req, request_vclock[min_req], granted_out, request_vclock[granted_out])) {
        /* advise current grantee that they should cancel */
        WHEN_DEBUG(fprintf(stderr, "suggest cancel %d at %d\n", granted_out, request_vclock[granted_out]));
        send_msg(granted_out, MAYBE_CANCEL, sizeof(int), &request_vclock[granted_out]);
      }
    }
  }
}

void cooperate()
{
  int from, tag, sz;
  void *data;

  if (me != REFEREE_SITE) {
    if (have_lock) {
      /* Since we cooperate after moving, release the lock here. */
      WHEN_DEBUG(fprintf(stderr, "release possession\n"));
      have_lock = 0;
      vclock = (vclock & ~(HOUR_HAND_INC - 1)) + HOUR_HAND_INC;
      grant_in_count = 0;
      request_vclock[me] = 0;
      broadcast_msg(RELEASE_STATE, sizeof(s), &s);
      check_grant();
    }
    if (!granted_out && !request_vclock[me]) {
      /* Issue a request. */
      WHEN_DEBUG(fprintf(stderr, "requesting %d\n", vclock));
      broadcast_msg(WANT_STATE, sizeof(int), &vclock);
      request_vclock[me] = vclock;
    }
  }

  data = recv_msg(&from, &tag, &sz, 0.0);
  if (!data) return;

  switch (tag) {
  case WANT_STATE:
    {
      int vc = *(int *)data;
      /* Site `from' requests the lock at time `vc'. */
      
      if (!vc || vc > request_vclock[from]) {
        /* Site `from' wants to cancel its earlier request: */
        WHEN_DEBUG(fprintf(stderr, "revising %d to %d\n", from, vc));
        request_vclock[from] = vc;
        if (granted_out == from)
          granted_out = 0;
      }

      if (request_vclock[me]) {
        /* We have previously issued a request... */
        if (!lt(from, vc, me, vclock)) {
          /* ... but someone requested with a time stamp later than
             ours; maybe they just joined, so repeat our request back to
             them */
          send_msg(from, WANT_STATE, sizeof(int), &request_vclock[me]);
        }
      }
        
      check_grant();
    }
    break;
  case GRANT_STATE:
    WHEN_DEBUG(fprintf(stderr, "given at %d (%d)\n", *(int *)data, grant_in_count));
    if (*(int *)data == request_vclock[me]) {
      grant_in_count++;
      if (grant_in_count > (NUM_SITES / 2)) {
        WHEN_DEBUG(fprintf(stderr, "took possession\n"));
        have_lock = 1;
      }
    }
    break;
  case RELEASE_STATE:
    memcpy(&s, data, sizeof(s));
    request_vclock[from] = 0;
    if (granted_out == from)
      granted_out = 0;
    WHEN_DEBUG(fprintf(stderr, "lock released by %d\n", from));
    check_grant();
    break;
  case MAYBE_CANCEL:
    if (*(int *)data == request_vclock[me]) {
      /* We have been granted a lock from site A (= `from'), but site
         A now thinks that site B should have been given the lock.
         Issue a cancel by incrementing our clock a small amount.  We
         never increment by small amounts to erach a whole "hour",
         because there are only so many ways to reorder messages (so
         that site A changes it mind) with a small number of players. */
      vclock++;
      WHEN_DEBUG(fprintf(stderr, "re-requesting at %d\n", vclock));
      broadcast_msg(WANT_STATE, sizeof(int), &vclock);
      request_vclock[me] = vclock;
      grant_in_count = 0;
    }
    break;
  default:
    WHEN_DEBUG(fprintf(stderr, "unexpected message %d\n", tag));
    abort();
  }
}
