#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include "dsm.h"

/**********************************************************************/

static dsm_sema_t m;
static dsm_sema_t stat_m;

static void lock_field()
{
  dsm_sema_wait(&m);
}

static void unlock_field()
{
  dsm_sema_signal(&m);
}

static void lock_stats()
{
  dsm_sema_wait(&stat_m);
}

static void unlock_stats()
{
  dsm_sema_signal(&stat_m);
}

/**********************************************************************/
/* Random-number generator                                            */

/* This is a stripped-down version of random.c as distributed with
   FreeBSD 2.2 */

#define	DEG		31
#define	SEP		3

typedef struct rand_state_t {
  short fpos, rpos;
  int state[DEG];
} rand_state_t;

static int bsd_rand(int n, rand_state_t *r)
{
  int i;

  r->state[r->fpos] += r->state[r->rpos];
  i = (r->state[r->fpos] >> 1) & 0x7fffffff; /* chucking least random bit */
  if (++(r->fpos) >= DEG) {
    r->fpos = 0;
    r->rpos++;
  } else if (++(r->rpos) >= DEG)
    r->rpos = 0;

  return i % n;
}

static void bsd_srand(int x, rand_state_t *r)
{
  register int i;

  r->state[0] = x & 0x7fffffff;
  for (i = 1; i < DEG; i++) {
    register long hi, lo, v;
  
    v = r->state[i - 1];
    hi = v / 127773;
    lo = v % 127773;
    v = 16807 * lo - 2836 * hi;
    if (v <= 0)
      v += 0x7fffffff;
    r->state[i] = v;
  }
  r->fpos = SEP;
  r->rpos = 0;
  for (i = 0; i < 10 * DEG; i++) {
    (void)bsd_rand(1, r);
  }
}

/**********************************************************************/
/* Game configuration                                                 */

/* End game after this many moves: */
#define MAX_MOVES 5000

/* Show field roughly once per epoch: */
#define MOVES_PER_EPOCH (MAX_MOVES / 10)

/* Counters: */
static int SHARED(moves) = 0;
static int SHARED(epochs) = 0;

static dsm_sema_t epoch_sema;

/* Game stats (per team): */
static int SHARED(passes)[2];
static int SHARED(steals)[2];

/* Field size: */
#define HSIZE 76
#define VSIZE 20
static struct player_t *SHARED(field)[VSIZE][HSIZE];

#define NUM_PLAYERS 4
#define NUM_BALLS 2

#define RED_START 1
#define BLUE_START (RED_START + NUM_PLAYERS)

static const int RED_TEAM = 0;
static const int BLUE_TEAM = 1;

typedef struct player_t {
  int rank;
  int team;
  int field_x, field_y;
  struct ball_t *ball; /* NULL => not carrying a ball */
  rand_state_t r;
  dsm_sema_t m;
} player_t;

typedef struct ball_t {
  int field_x, field_y;
  player_t *carried_by; /* NULL => directly on field */
} ball_t;

static player_t SHARED(red)[NUM_PLAYERS];
static player_t SHARED(blue)[NUM_PLAYERS];
static ball_t SHARED(ball)[NUM_BALLS];

static void lock_player(player_t *p)
{
  dsm_sema_wait(&p->m);
}

static void unlock_player(player_t *p)
{
  dsm_sema_signal(&p->m);
}

/**********************************************************************/
/* Ball-transfer contest                                              */

static int contest(player_t *p1, player_t *p2)
/* Generate a bunch of random numbers to determine whether a
   pass/steal succeeds. (Simulates a more interesting calculation that
   a real game might use.) */
{
  int c1 = 0, c2 = 0;

  while ((c1 < 10000) && (c2 < 10000)) {
    if (bsd_rand(100, &p1->r) > bsd_rand(100, &p1->r))
      c1++;
    else
      c2++;
  }

  return (c1 > c2);
}

/**********************************************************************/
/* The player thread's procedure                                      */

void *run_player(void *_p) 
{
  player_t *p = (player_t *)_p, *p2;

  while (1) {
    int x, y;

    x = p->field_x;
    y = p->field_y;
    switch (bsd_rand(4, &p->r)) {
    case 0: x++; break;
    case 1: --x; break;
    case 2: y++; break;
    case 3: y--; break;
    }
     
    if ((x >= 0) && (x < HSIZE)
        && (y >= 0) && (y < VSIZE)) {
      lock_field();
      if (SHARED(moves) >= MAX_MOVES) {
        unlock_field();
        return NULL;
      }
      SHARED(moves)++;
      if ((SHARED(moves) % MOVES_PER_EPOCH) == 0) {
        SHARED(epochs)++;
        dsm_sema_signal(&epoch_sema);
      }

      p2 = SHARED(field)[y][x];
      if (!p2) {
        /* move */
        lock_player(p);
        SHARED(field)[p->field_y][p->field_x] = NULL;
        SHARED(field)[y][x] = p;
        p->field_x = x;
        p->field_y = y;
        if (p->ball) {
          p->ball->field_x = x;
          p->ball->field_y = y;
        }
        unlock_player(p);
        unlock_field();
      } else {
        if (p->rank < p2->rank) {
          lock_player(p);
          lock_player(p2);
        } else {
          lock_player(p2);
          lock_player(p);
        }
        unlock_field();
        if (p->ball && (p->team == p2->team) && !p2->ball) {
          /* try to pass */
          if (contest(p, p2)) {
            /* a pass */
            p2->ball = p->ball;
            p->ball->field_x = x;
            p->ball->field_y = y;
            p->ball->carried_by = p2;
            p->ball = NULL;
            lock_stats();
            SHARED(passes)[p->team]++;
            unlock_stats();
          }
        } else if (!p->ball && (p->team != p2->team) && p2->ball) {
          /* try to steal... */
          if (contest(p, p2)) {
            /* a steal */
            p->ball = p2->ball;
            p2->ball = NULL;
            p->ball->field_x = p->field_x;
            p->ball->field_y = p->field_y;
            p->ball->carried_by = p;
            lock_stats();
            SHARED(steals)[p->team]++;
            unlock_stats();
          }
        } else {
          /* bump */
        }
        unlock_player(p2);
        unlock_player(p);
      }
    }
  }
}

/**********************************************************************/
/* Showing the field                                                  */

static void show_field()
{
  int i, j;

  printf("\nMoves: %d  Red passes+steals: %d+%d  Blue passes+steals: %d+%d\n", 
         SHARED(moves), 
         SHARED(passes)[RED_TEAM], SHARED(steals)[RED_TEAM], 
         SHARED(passes)[BLUE_TEAM], SHARED(steals)[BLUE_TEAM]);
  for (i = 0; i < VSIZE; i++) {
    for (j = 0; j < HSIZE; j++) {
      player_t *p = SHARED(field)[i][j];
      if (p) {
        char c;
        c = ((p->team == RED_TEAM) ? 'r' : 'b');
        if (p->ball)
          c = toupper(c);
        printf("%c", c);
        /* Sanity check: */
        if (p->ball)
          if (p->ball->carried_by != p) {
            fprintf(stderr, "Player has ball, but ball doesn't refer to player\n");
            abort();
          }
      } else
        printf("_");
    }
    printf("\n");
  }

  /* More sanity checks: */
  for (i = 0; i < NUM_PLAYERS; i++) {
    if (SHARED(field)[SHARED(red)[i].field_y][SHARED(red)[i].field_x] != &SHARED(red)[i]) {
      fprintf(stderr, "A red player's position doesn't match the field\n");
      abort();
    }
    if (SHARED(field)[SHARED(blue)[i].field_y][SHARED(blue)[i].field_x] != &SHARED(blue)[i]) {
      fprintf(stderr, "A blue player's position doesn't match the field\n");
      abort();
    }
  }
  for (i = 0; i < NUM_BALLS; i++) {
    if (SHARED(ball)[i].carried_by->ball != &SHARED(ball)[i]) {
      fprintf(stderr, "Ball's carrier does not refer to ball\n");
      abort();
    }
    if ((SHARED(ball)[i].field_x != SHARED(ball)[i].carried_by->field_x)
        || (SHARED(ball)[i].field_y != SHARED(ball)[i].carried_by->field_y)) {
      fprintf(stderr, "Ball is not at its carrier's position\n");
      abort();
    }
  }
}

/**********************************************************************/
/* Main                                                               */

static dsm_sema_t init_sema;

void site_init()
{
  int i;

  INIT_SHARED(field);
  INIT_SHARED(passes);
  INIT_SHARED(steals);
  INIT_SHARED(red);
  INIT_SHARED(blue);
  INIT_SHARED(ball);
  INIT_SHARED(moves);
  INIT_SHARED(epochs);

  dsm_sema_init(&m, 1);
  dsm_sema_init(&stat_m, 1);

  dsm_sema_init(&init_sema, 0);
  dsm_sema_init(&epoch_sema, 0);

  for (i = 0; i < NUM_PLAYERS; i++) {
    dsm_sema_init(&SHARED(red)[i].m, 1);
    dsm_sema_init(&SHARED(blue)[i].m, 1);
  }
}

void site_main(int id)
{
  int i;
  int prev_shown_epochs = 0;

  if (id == 0) {
    /* Put players on the field: */
    int mix = (int)time(NULL);

    for (i = 0; i < NUM_PLAYERS; i++) {
      SHARED(red)[i].rank = 2*i;
      SHARED(red)[i].field_x = i;
      SHARED(red)[i].field_y = 0;
      SHARED(red)[i].team = RED_TEAM;
      SHARED(red)[i].ball = NULL;
      bsd_srand(mix + i, &SHARED(red)[i].r);

      SHARED(field)[0][i] = &SHARED(red)[i];

      SHARED(blue)[i].rank = 2*i + 1;
      SHARED(blue)[i].field_x = i;
      SHARED(blue)[i].field_y = VSIZE-1;
      SHARED(blue)[i].team = BLUE_TEAM;
      SHARED(blue)[i].ball = NULL;
      bsd_srand(mix - i, &SHARED(blue)[i].r);

      SHARED(field)[VSIZE-1][i] = &SHARED(blue)[i];
    }

    /* Give one ball to each team: */
    SHARED(ball)[0].carried_by = &SHARED(red)[0];
    SHARED(ball)[1].carried_by = &SHARED(blue)[0];
    for (i = 0; i < NUM_BALLS; i++) {
      SHARED(ball)[i].carried_by->ball = &SHARED(ball)[i];
      SHARED(ball)[i].field_x = SHARED(ball)[i].carried_by->field_x;
      SHARED(ball)[i].field_y = SHARED(ball)[i].carried_by->field_y;
    }
    for (i = 0; i < NUM_PLAYERS; i++) {
      dsm_sema_signal(&init_sema);
      dsm_sema_signal(&init_sema);
    }
  } else {
    dsm_sema_wait(&init_sema);
  }

  /* Start the players: */
  for (i = 0; i < NUM_PLAYERS; i++) {
    if (id == RED_START + i)
      run_player(&SHARED(red)[i]);
    if (id == BLUE_START + i)
      run_player(&SHARED(blue)[i]);
  }

  if (id == 0) {
    /* Update field display occassionally: */
    while (1) {
      lock_field();
      if (SHARED(moves) >= MAX_MOVES) {
        unlock_field();
        break;
      }
      if (SHARED(epochs) > prev_shown_epochs) {
        for (i = 0; i < NUM_PLAYERS; i++) {
          lock_player(&SHARED(red)[i]);
          lock_player(&SHARED(blue)[i]);
        }
        lock_stats();
        show_field();
        unlock_stats();
        for (i = 0; i < NUM_PLAYERS; i++) {
          unlock_player(&SHARED(red)[i]);
          unlock_player(&SHARED(blue)[i]);
        }
        prev_shown_epochs = SHARED(epochs);
        unlock_field();
      } else {
        unlock_field();
        dsm_sema_wait(&epoch_sema);
      }
    }

    /* Game over. Wait for players to stop, then show field
       one last time (just in case). */

    printf("Game over\n");

    show_field();
  }
}

