#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include "dsm.h"
#include "network_dsm.h"

static int fd;

/* Blocks of shared memory: */
static char *mems[NUM_MEMS];
static int mem_sizes[NUM_MEMS];

/* A copy of each block, used to detect changes: */
static char *old_mems[NUM_MEMS];

void *dsm_malloc(int size)
{
  int buffer[2], n;
  void *addr;

  buffer[0] = MSG_ALLOC_MEMORY;
  buffer[1] = size;

  /* Rquest memory id for `size' bytes: */
  write_all(fd, buffer, 2 * sizeof(int));
  read_all(fd, buffer, sizeof(int));

  /* n is memory id: */
  n = buffer[0];

  /* Get address to use for memory: */
  read_all(fd, &addr, sizeof(void*));
  mems[n] = (char *)mmap(addr, size, PROT_READ | PROT_WRITE, 
                         MAP_ANON | MAP_PRIVATE, -1, 0);
  /* Send back the address that we use: */
  write_all(fd, &(mems[n]), sizeof(void*));

  /* Initialize memory and the copy to use for detecting changes: */
  memset(mems[n], 0, size);
  old_mems[n] = (char *)malloc(size);
  memcpy(old_mems[n], mems[n], size);

  mem_sizes[n] = size;

  return mems[n];
}

/* **************************************** */

void dsm_sema_init(dsm_sema_t *t, int state)
{
  /* Get a semaphore id from the server: */
  int buffer[2];

  buffer[0] = MSG_ALLOC_SEMA;
  buffer[1] = state;

  write_all(fd, buffer, 2 * sizeof(int));

  read_all(fd, buffer, sizeof(int));

  *t = buffer[0];
}

void dsm_sema_wait(dsm_sema_t *t)
{
  int buffer[2], i, k, size;
  char *mask, *data;

  /* Wait on a semaphore; server replies when we are allowed to
     continue: */
  buffer[0] = MSG_WAIT_SEMA;
  write_all(fd, buffer, sizeof(int));
  buffer[0] = *t;
  write_all(fd, buffer, sizeof(int));
  read_all(fd, buffer, sizeof(int));

  /* Download updates to memory from the server: */
  for (i = 0; i < NUM_MEMS; i++) {
    if (mems[i]) {
      size = mem_sizes[i];
      data = (char *)malloc(size);
      mask = (char *)malloc(size);

      buffer[0] = MSG_READ;
      write_all(fd, buffer, sizeof(int));
      write_all(fd, &mems[i], sizeof(void*));
      write_all(fd, &size, sizeof(int));

      read_all(fd, data, size);
      read_all(fd, mask, size);

      /* For each byte changed by other sites, update the memory at
         this site (includng the copy, since the change doesn't come
         from here): */
      for (k = 0; k < size; k++) {
        if (mask[k]) {
          mems[i][k] = data[k];
          old_mems[i][k] = data[k];
        }
      }
      
      free(data);
      free(mask);
    }
  }
}

void dsm_sema_signal(dsm_sema_t *t)
{
  int buffer[2], i, k, size;
  char *mask;

  /* Send updates to memory to the server: */
  for (i = 0; i < NUM_MEMS; i++) {
    if (mems[i]) {
      size = mem_sizes[i];
      mask = (char *)malloc(size);

      /* Compute which bytes we changed based on the memory copy (and
         set the memory copy to match current memory, since we are
         sending the update). */
      for (k = 0; k < size; k++) {
        mask[k] = !(old_mems[i][k] == mems[i][k]);
        old_mems[i][k] = mems[i][k];
      }

      buffer[0] = MSG_WRITE;
      write_all(fd, buffer, sizeof(int));
      write_all(fd, &mems[i], sizeof(void*));
      write_all(fd, &size, sizeof(int));

      write_all(fd, mems[i], size);
      write_all(fd, mask, size);

      free(mask);
    }
  }

  /* Finally, signal the semaphore */

  buffer[0] = MSG_SIGNAL_SEMA;
  write_all(fd, buffer, sizeof(int));
  buffer[0] = *t;
  write_all(fd, buffer, sizeof(int));
}

/* **************************************** */

void notify_init_done()
{
  /* Tell the server that we are done creating memory and
     semaphores: */
  int buffer[2];

  buffer[0] = MSG_INIT_DONE;
  buffer[1] = 0;

  write_all(fd, buffer, 2 * sizeof(int));
}

void notify_done()
{
  /* Tell the server that we are done: */
  int buffer[1];

  buffer[0] = MSG_DONE;

  write_all(fd, buffer, sizeof(int));
}

/* **************************************** */

int main(int argc, char **argv)
{
  int id = atoi(argv[1]);

  fd = start_client(argv[2], argv[3], id);

  site_init();
  notify_init_done();
  
  /* The main program: */
  site_main(id);

  notify_done();

  close(fd);

  return 1;
}
