#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <rpc/rpc.h>
#include "field.h"
#include "const.h"
#include "rand.h"

/**********************************************************************/
/* Game configuration                                                 */

/* Show field roughly once per epoch: */
#define MOVES_PER_EPOCH (MAX_MOVES / 10)

/* Counters: */
static int moves = 0;

/* Game stats (per team): */
static int passes[2] = { 0, 0 };
static int steals[2] = { 0, 0 };

/* Field size: */
static struct player_t *field[VSIZE][HSIZE];

#define NUM_PLAYERS 4
#define NUM_BALLS 2

static const int RED_TEAM = 0;
static const int BLUE_TEAM = 1;

typedef struct player_t {
  int id;
  int team;
  int field_x, field_y;
  struct ball_t *ball; /* NULL => not carrying a ball */
} player_t;

typedef struct ball_t {
  int field_x, field_y;
  player_t *carried_by; /* NULL => directly on field */
} ball_t;

static player_t red[NUM_PLAYERS];
static player_t blue[NUM_PLAYERS];
static ball_t ball[NUM_BALLS];

#define RED_START 1
#define BLUE_START (NUM_PLAYERS + RED_START)

/**********************************************************************/
/* Showing the field                                                  */

static void do_init_field()
{
  static int init_done = 0;
  int i;

  if (init_done)
    return;

  /* Put players on the field: */
  for (i = 0; i < NUM_PLAYERS; i++) {
    red[i].id = i + RED_START;
    red[i].field_x = i;
    red[i].field_y = 0;
    red[i].team = RED_TEAM;
    red[i].ball = NULL;

    field[0][i] = &red[i];

    blue[i].id = i + BLUE_START;
    blue[i].field_x = i;
    blue[i].field_y = VSIZE-1;
    blue[i].team = BLUE_TEAM;
    blue[i].ball = NULL;

    field[VSIZE-1][i] = &blue[i];
  }

  /* Give one ball to each team: */
  ball[0].carried_by = &red[0];
  ball[1].carried_by = &blue[0];
  for (i = 0; i < NUM_BALLS; i++) {
    ball[i].carried_by->ball = &ball[i];
    ball[i].field_x = ball[i].carried_by->field_x;
    ball[i].field_y = ball[i].carried_by->field_y;
  }

  init_done = 1;
}

/**********************************************************************/
/* Showing the field                                                  */

static void show_field()
{
  int i, j;

  printf("\nMoves: %d  Red passes+steals: %d+%d  Blue passes+steals: %d+%d\n", 
         moves, 
         passes[RED_TEAM], steals[RED_TEAM], 
         passes[BLUE_TEAM], steals[BLUE_TEAM]);
  for (i = 0; i < VSIZE; i++) {
    for (j = 0; j < HSIZE; j++) {
      player_t *p = field[i][j];
      if (p) {
        char c;
        c = ((p->team == RED_TEAM) ? 'r' : 'b');
        if (p->ball)
          c = toupper(c);
        printf("%c", c);
        /* Sanity check: */
        if (p->ball)
          if (p->ball->carried_by != p) {
            fprintf(stderr, "Player has ball, but ball doesn't refer to player\n");
            abort();
          }
      } else
        printf("_");
    }
    printf("\n");
  }

  /* More sanity checks: */
  for (i = 0; i < NUM_PLAYERS; i++) {
    if (field[red[i].field_y][red[i].field_x] != &red[i]) {
      fprintf(stderr, "A red player's position doesn't match the field\n");
      abort();
    }
    if (field[blue[i].field_y][blue[i].field_x] != &blue[i]) {
      fprintf(stderr, "A blue player's position doesn't match the field\n");
      abort();
    }
  }
  for (i = 0; i < NUM_BALLS; i++) {
    if (ball[i].carried_by->ball != &ball[i]) {
      fprintf(stderr, "Ball's carrier does not refer to ball\n");
      abort();
    }
    if ((ball[i].field_x != ball[i].carried_by->field_x)
        || (ball[i].field_y != ball[i].carried_by->field_y)) {
      fprintf(stderr, "Ball is not at its carrier's position\n");
      abort();
    }
  }
}

static void inc_moves()
{
  moves++;
  if ((moves % MOVES_PER_EPOCH) == 0) {
    show_field();
  }
}

/**********************************************************************/
/* Changing the field                                                 */

static int ok = 1;

static player_t *get_player(int p_id)
{
  return (p_id >= BLUE_START 
          ? &blue[p_id - BLUE_START]
          : &red[p_id - RED_START]);
}

void *init_field_1_svc(int p_id, struct svc_req *r)
{
  do_init_field();
  return &ok;
}

int *get_moves_1_svc(int p_id, struct svc_req *r)
{
  return &moves;
}

int *get_player_x_1_svc(int p_id, struct svc_req *r)
{
  return &get_player(p_id)->field_x;
}

int *get_player_y_1_svc(int p_id, struct svc_req *r)
{
  return &get_player(p_id)->field_y;
}

int *get_player_ball_1_svc(int p_id, struct svc_req *r)
{
  static int ball_id;
  player_t *p = get_player(p_id);

  if (p->ball) {
    if (p->ball == &ball[0])
      ball_id = 1;
    else
      ball_id = 2;
  } else
    ball_id = 0;

  return &ball_id;
}

int *get_player_team_1_svc(int p_id, struct svc_req *r)
{
  static int team_id;
  
  if (p_id >= BLUE_START)
    team_id = BLUE_TEAM;
  else
    team_id = RED_TEAM;

  return &team_id;
}

int *get_field_player_1_svc(int y, int x, struct svc_req *r)
{
  static int player_id;

  if (field[y][x])
    player_id = field[y][x]->id;
  else
    player_id = 0;

  return &player_id;
}

void *move_player_1_svc(int p_id, int y, int x, struct svc_req *r)
{
  player_t *p = get_player(p_id);

  field[p->field_y][p->field_x] = NULL;
  field[y][x] = p;
  p->field_x = x;
  p->field_y = y;
  if (p->ball != NULL) {
    p->ball->field_x = x;
    p->ball->field_y = y;
  }
  
  inc_moves();

  return &ok;
}

void *move_ball_1_svc(int p_id, int p2_id, int pass, struct svc_req *r)
{
  player_t *p = get_player(p_id);
  player_t *p2 = get_player(p2_id);

  p2->ball = p->ball;
  p->ball->field_x = p2->field_x;
  p->ball->field_y = p2->field_y;
  p->ball->carried_by = p2;
  p->ball = NULL;
  if (pass)
    passes[p->team]++;
  else
    steals[p2->team]++;
  
  inc_moves();

  return &ok;
}
