#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <rpc/rpc.h>
#include "field.h"
#include "const.h"
#include "rand.h"

/**********************************************************************/
/* Game configuration                                                 */

/* Show field roughly once per epoch: */
#define MOVES_PER_EPOCH (MAX_MOVES / 10)

/* Counters: */
static int moves = 0;

/* Game stats (per team): */
static int passes[2] = { 0, 0 };
static int steals[2] = { 0, 0 };

/* Field size: */
static struct player_t *field[VSIZE][HSIZE];

#define NUM_PLAYERS 4
#define NUM_BALLS 2

static const int RED_TEAM = 0;
static const int BLUE_TEAM = 1;

typedef struct player_t {
  int id;
  int team;
  int field_x, field_y;
  struct ball_t *ball; /* NULL => not carrying a ball */
} player_t;

typedef struct ball_t {
  int field_x, field_y;
  player_t *carried_by; /* NULL => directly on field */
} ball_t;

static player_t red[NUM_PLAYERS];
static player_t blue[NUM_PLAYERS];
static ball_t ball[NUM_BALLS];

#define RED_START 1
#define BLUE_START (NUM_PLAYERS + RED_START)

/**********************************************************************/
/* Transaction logs                                                   */

typedef struct log_entry_t {
  void *addr;
  int is_int;
  union {
    int i;
    void *p;
  } orig_val;
  union {
    int i;
    void *p;
  } new_val;
} log_entry_t;

#define MAX_LOG_ENTRIES 32

static int aborts;

static log_entry_t log[NUM_PLAYERS + BLUE_START][MAX_LOG_ENTRIES];
static int log_size[NUM_PLAYERS + BLUE_START];

static void write_int(int p_id, int *ip, int v)
{
  int i;

  for (i = log_size[p_id]; i--; ) {
    if (log[p_id][i].addr == ip) {
      log[p_id][i].new_val.i = v;
      return;
    }
  }

  i = log_size[p_id]++;
  if (i > MAX_LOG_ENTRIES) {
    fprintf(stderr, "log-entry overflow\n");
    abort();
  }

  log[p_id][i].addr = ip;
  log[p_id][i].is_int = 1;
  log[p_id][i].orig_val.i = *ip;
  log[p_id][i].new_val.i = v;
}

static void write_ptr(int p_id, void *ip, void *v)
{
  int i;

  for (i = log_size[p_id]; i--; ) {
    if (log[p_id][i].addr == ip) {
      log[p_id][i].new_val.p = v;
      return;
    }
  }

  i = log_size[p_id]++;
  log[p_id][i].addr = ip;
  log[p_id][i].is_int = 0;
  log[p_id][i].orig_val.p = *(void **)ip;
  log[p_id][i].new_val.p = v;
}

static int read_int(int p_id, int *ip)
{
  int i;
  int v;

  for (i = log_size[p_id]; i--; ) {
    if (log[p_id][i].addr == ip) {
      return log[p_id][i].new_val.i;
    }
  }

  v = *ip;

  /* Log read as a write of current value: */
  write_int(p_id, ip, v);

  return v;
}
  
static void *read_ptr(int p_id, void *ip)
{
  int i;
  void *v;

  for (i = log_size[p_id]; i--; ) {
    if (log[p_id][i].addr == ip) {
      return log[p_id][i].new_val.p;
    }
  }

  v = *(void **)ip;

  /* Log read as a write of current value: */
  write_ptr(p_id, ip, v);

  return v;
}

/**********************************************************************/
/* Showing the field                                                  */

static void do_init_field()
{
  static int init_done = 0;
  int i;

  if (init_done)
    return;

  /* Put players on the field: */
  for (i = 0; i < NUM_PLAYERS; i++) {
    red[i].id = i + RED_START;
    red[i].field_x = i;
    red[i].field_y = 0;
    red[i].team = RED_TEAM;
    red[i].ball = NULL;

    field[0][i] = &red[i];

    blue[i].id = i + BLUE_START;
    blue[i].field_x = i;
    blue[i].field_y = VSIZE-1;
    blue[i].team = BLUE_TEAM;
    blue[i].ball = NULL;

    field[VSIZE-1][i] = &blue[i];
  }

  /* Give one ball to each team: */
  ball[0].carried_by = &red[0];
  ball[1].carried_by = &blue[0];
  for (i = 0; i < NUM_BALLS; i++) {
    ball[i].carried_by->ball = &ball[i];
    ball[i].field_x = ball[i].carried_by->field_x;
    ball[i].field_y = ball[i].carried_by->field_y;
  }

  init_done = 1;
}

/**********************************************************************/
/* Showing the field                                                  */

static void show_field()
{
  int i, j;

  printf("\nMoves: %d  Red passes+steals: %d+%d  Blue passes+steals: %d+%d  Aborts: %d\n", 
         moves, 
         passes[RED_TEAM], steals[RED_TEAM], 
         passes[BLUE_TEAM], steals[BLUE_TEAM],
         aborts);
  for (i = 0; i < VSIZE; i++) {
    for (j = 0; j < HSIZE; j++) {
      player_t *p = field[i][j];
      if (p) {
        char c;
        c = ((p->team == RED_TEAM) ? 'r' : 'b');
        if (p->ball)
          c = toupper(c);
        printf("%c", c);
        /* Sanity check: */
        if (p->ball)
          if (p->ball->carried_by != p) {
            fprintf(stderr, "Player has ball, but ball doesn't refer to player\n");
            abort();
          }
        if ((p->field_x != j) || (p->field_y != i)) {
          fprintf(stderr, "Field's player is not at the field position\n");
          abort();
        }
      } else
        printf("_");
    }
    printf("\n");
  }

  /* More sanity checks: */
  for (i = 0; i < NUM_PLAYERS; i++) {
    if (field[red[i].field_y][red[i].field_x] != &red[i]) {
      fprintf(stderr, "A red player's position doesn't match the field\n");
      abort();
    }
    if (field[blue[i].field_y][blue[i].field_x] != &blue[i]) {
      fprintf(stderr, "A blue player's position doesn't match the field\n");
      abort();
    }
  }
  for (i = 0; i < NUM_BALLS; i++) {
    if (ball[i].carried_by->ball != &ball[i]) {
      fprintf(stderr, "Ball's carrier does not refer to ball\n");
      abort();
    }
    if ((ball[i].field_x != ball[i].carried_by->field_x)
        || (ball[i].field_y != ball[i].carried_by->field_y)) {
      fprintf(stderr, "Ball is not at its carrier's position\n");
      abort();
    }
  }
}

static void inc_moves(int p_id)
{
  write_int(p_id, &moves, read_int(p_id, &moves) + 1);
}

/**********************************************************************/
/* Changing the field                                                 */

static int ok = 1;
static int not_ok = 1;

static player_t *get_player(int p_id)
{
  return (p_id >= BLUE_START 
          ? &blue[p_id - BLUE_START]
          : &red[p_id - RED_START]);
}

void *init_field_1_svc(int p_id, struct svc_req *r)
{
  do_init_field();
  return &ok;
}

int *get_player_x_1_svc(int p_id, struct svc_req *r)
{
  static int x;
  x = read_int(p_id, &get_player(p_id)->field_x);
  return &x;
}

int *get_player_y_1_svc(int p_id, struct svc_req *r)
{
  static int y;
  y = read_int(p_id, &get_player(p_id)->field_y);
  return &y;
}

int *get_player_ball_1_svc(int p_id, struct svc_req *r)
{
  static int ball_id;
  player_t *p = get_player(p_id);
  
  if (read_ptr(p_id, &p->ball)) {
    if (read_ptr(p_id, &p->ball) == (void *)&ball[0])
      ball_id = 1;
    else
      ball_id = 2;
  } else
    ball_id = 0;

  return &ball_id;
}

int *get_player_team_1_svc(int p_id, struct svc_req *r)
{
  static int team_id;
  
  if (p_id >= BLUE_START)
    team_id = BLUE_TEAM;
  else
    team_id = RED_TEAM;

  return &team_id;
}

int *get_field_player_1_svc(int p_id, int y, int x, struct svc_req *r)
{
  player_t *p;
  static int player_id;

  p = (player_t *)read_ptr(p_id, &field[y][x]);

  if (p)
    player_id = p->id;
  else
    player_id = 0;

  return &player_id;
}

void *move_player_1_svc(int p_id, int y, int x, struct svc_req *r)
{
  player_t *p = get_player(p_id);
  ball_t *b;

  write_ptr(p_id, &field[p->field_y][p->field_x], NULL);
  write_ptr(p_id, &field[y][x], p);
  write_int(p_id, &p->field_x, x);
  write_int(p_id, &p->field_y, y);

  b = (ball_t *)read_ptr(p_id, &p->ball);
  if (b != NULL) {
    write_int(p_id, &b->field_x, x);
    write_int(p_id, &b->field_y, y);
  }

  inc_moves(p_id);

  return &ok;
}

void *move_ball_1_svc(int p1_id, int p2_id, int pass, struct svc_req *r)
{
  player_t *p1 = get_player(p1_id);
  player_t *p2 = get_player(p2_id);
  ball_t *b;
  int p_id = (pass ? p1_id : p2_id);

  write_ptr(p_id, &p2->ball, p1->ball);

  b = (ball_t *)read_ptr(p_id, &p1->ball);
  write_int(p_id, &b->field_x, p2->field_x);
  write_int(p_id, &b->field_y, p2->field_y);
  write_ptr(p_id, &b->carried_by, p2);
  write_ptr(p_id, &p1->ball, NULL);

  if (pass)
    write_int(p_id, &passes[p1->team], 
              read_int(p_id, &passes[p1->team]) + 1);
  else
    write_int(p_id, &steals[p1->team], 
              read_int(p_id, &steals[p1->team]) + 1);
  
  inc_moves(p_id);

  return &ok;
}


int *start_transaction_1_svc(int p_id, struct svc_req *r)
{
  log_size[p_id] = 0;

  return &ok;
}

int *should_abort_1_svc(int p_id, struct svc_req *r)
{
  static int ret;
  int i;

  for (i = log_size[p_id]; i--; ) {
    if (log[p_id][i].is_int) {
      if (log[p_id][i].orig_val.i != *(int *)log[p_id][i].addr) {
        aborts++;
        ret = 1;
        return &ret;
      }
    } else {
      if (log[p_id][i].orig_val.p != *(void **)log[p_id][i].addr) {
        aborts++;
        ret = 1;
        return &ret;
      }
    }
  }

  ret = 0;
  return &ret;
}

int *end_transaction_1_svc(int p_id, struct svc_req *r)
{
  int i;
  
  if (*should_abort_1_svc(p_id, r))
    return &not_ok;

  for (i = log_size[p_id]; i--; ) {
    if (log[p_id][i].is_int) {
      *(int *)log[p_id][i].addr = log[p_id][i].new_val.i;
    } else {
      *(void **)log[p_id][i].addr = log[p_id][i].new_val.p;
    }
  }

  if ((moves % MOVES_PER_EPOCH) == 0) {
    show_field();
  }

  return &ok;
}
